% main_postestimation_filtering.m
% 
% computes postestimation filtering exercises 
% 
% Estimating Macroeconomic Models of Financial Crises: An Endogenous Regime-Switching Approach*
%   Gianluca Benigno, Andrew Foerster, Christopher Otrok, Alessandro Rebucci
% 
% Updated July 2024
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %


% -- Clear Workspace -- %
clc;
clear;
close all;
restoredefaultpath;

% -- Options -- %
addpath('../Solver - SV')
addpath(genpath('../Utilities'));
GLOBALS = globals();

% - Data -- %
y               = GLOBALS.y;
dates           = GLOBALS.dates;
rrdates         = GLOBALS.rrdates;

% -- Parameter Vector -- %
qm          = getparams(GLOBALS);



%% -- Calculate Filtered and Smoothed Estimates -- %%
% -- Setup -- %
kf_filtering_level = 1;
NX      = GLOBALS.estimation.NX;
NY      = GLOBALS.estimation.NY;

% -- Get Filtered Estimates -- %
[Posterior, Filters, Solution] = endo_ms_lik_2nd_kim_nelson_SV(qm,y,kf_filtering_level,1,0,GLOBALS);

% -- Average Filtered and Smoothed States -- %
AV_att = NaN(NX,size(y,2));     % Average Filtered States
AV_atT  = NaN(NX,size(y,2));    % Average Smoothed States
for i = 1:size(y,2)
    AV_att(:,i) = Filters.PAItt(1,i)*Filters.att(:,i,1) + Filters.PAItt(2,i)*Filters.att(:,i,2) + Filters.PAItt(3,i)*Filters.att(:,i,3) + Filters.PAItt(4,i)*Filters.att(:,i,4);
    AV_atT(:,i) = Filters.PAItT(1,i)*Filters.atT(:,1,i) + Filters.PAItT(2,i)*Filters.atT(:,2,i) + Filters.PAItT(3,i)*Filters.atT(:,3,i) + Filters.PAItT(4,i)*Filters.atT(:,4,i);
end

% -- Observation Equation Selection Matrix -- %
D       = GLOBALS.estimation.D;

% -- Set Average Observables -- %
filtered_observables_average        = NaN(NY,size(y,2));
smoothed_observables_average        = NaN(NY,size(y,2));
for kk = 1:size(y,2)
    filtered_observables_average(:,kk) = D*AV_att(:,kk);	% Observable Average
    smoothed_observables_average(:,kk) = D*AV_atT(:,kk);    % Observable Average
end




%% -- Figure 3: Data and Model Estimates -- %%

figure(3);
for ii = 1:NY
    subplot(NY,1,ii);

    hold on
    bar(dates,rrdates*(1.2*max([y(ii,:) filtered_observables_average(ii,:) smoothed_observables_average(ii,:)])),1.02,'FaceColor',[0.75 0.75 0.75],'EdgeColor','none','ShowBaseLine','off')  
    bar(dates,rrdates*(1.2*min([y(ii,:) filtered_observables_average(ii,:) smoothed_observables_average(ii,:)])),1.02,'FaceColor',[0.75 0.75 0.75],'EdgeColor','none','ShowBaseLine','off')  
    h1 = plot(dates,y(ii,:),'k-','Linewidth',2);
    h2 = plot(dates,filtered_observables_average(ii,:),'r--','Linewidth',2);
    plot(dates,zeros(1,length(y)),'k--','Linewidth',1);
    hold off
    xticks(dates(1:4:end))
        datetick('x',17,'keepticks');    %#ok<*DATIC>
        ax = gca;
        labels = string(ax.XAxis.TickLabels);
        labels(1:5:end) = nan; 
        labels(2:5:end) = nan; 
        labels(3:5:end) = nan; 
        labels(4:5:end) = nan; 
        ax.XAxis.TickLabels = labels;    
        xlim([dates(1) dates(end)]);
        xtickangle(0)
        set(gca, 'Layer','top')
    axis tight
    set(gca, 'Layer','top')
    if ii == 1
        legend([h1,h2],'Data','Model','Orientation','Horizontal','Location','Southeast');
    end
    box on
    title(GLOBALS.obslist{ii})
end
print('-dpdf','..\Figures and Tables\Figure_3.pdf', '-fillpage')



%% -- Figure 4: Model Estimated Shocks -- %%


figure(4)
for ii = 1:GLOBALS.ne
    subplot(GLOBALS.ne,1,ii);
    hold on
    bar(dates, 4*rrdates,1.02,'FaceColor',[0.75 0.75 0.75],'EdgeColor','none','ShowBaseLine','off')  
    bar(dates,-4*rrdates,1.02,'FaceColor',[0.75 0.75 0.75],'EdgeColor','none','ShowBaseLine','off')  
    h2 = plot(dates,AV_atT(2*(GLOBALS.nx+GLOBALS.ny)+ii,:),'k-','Linewidth',2);
    plot(dates,zeros(1,length(y)),'k--','Linewidth',1);
    plot(dates,[-2;2]*ones(1,length(y)),'k--','Linewidth',1);
    hold off
    xticks(dates(1:4:end))
        datetick('x',17,'keepticks');    
        ax = gca;
        labels = string(ax.XAxis.TickLabels);
        labels(1:5:end) = nan; 
        labels(2:5:end) = nan; 
        labels(3:5:end) = nan; 
        labels(4:5:end) = nan; 
        ax.XAxis.TickLabels = labels;    
        xlim([dates(1) dates(end)]);
        xtickangle(0)
        set(gca, 'Layer','top')
    axis tight
    ylim([-4 4])
    set(gca, 'Layer','top')
    box on
    title(GLOBALS.shocklist{ii,2})
end
print('-dpdf','..\Figures and Tables\Figure_4.pdf', '-fillpage')



%% -- Figure 5: Mexico's Model-identfieded Crisis Episodes -- %%

figure(5);
subplot(2,1,1)
    title('Probability of Binding Regime and Reinhart-Rogoff Tally Index')
    hold on
    bar(dates,rrdates,1.02,'FaceColor',[0.75 0.75 0.75],'EdgeColor','none','ShowBaseLine','off');
    plot(dates,Filters.PAItT(3,1:end)+Filters.PAItT(4,1:end),'k-','Linewidth',2);
    plot(dates,Filters.PAItt(3,1:end)+Filters.PAItt(4,1:end),'b-','Linewidth',1.5);
    hold off
    xticks(dates(1:4:end))
        datetick('x',17,'keepticks');    
        ax = gca;
        labels = string(ax.XAxis.TickLabels);
        labels(1:5:end) = nan; 
        labels(2:5:end) = nan; 
        labels(3:5:end) = nan; 
        labels(4:5:end) = nan; 
        ax.XAxis.TickLabels = labels;    
        xlim([dates(1) dates(end)]);
        xtickangle(0)
    set(gca, 'Layer','top')
    axis tight
    box on
   
subplot(2,1,2)
    title('Probability of High Volatility Regime')
    hold on
    bar(dates,rrdates,1.02,'FaceColor',[0.75 0.75 0.75],'EdgeColor','none','ShowBaseLine','off');
    plot(dates,Filters.PAItT(2,1:end)+Filters.PAItT(4,1:end),'k-','Linewidth',2);
    plot(dates,Filters.PAItt(2,1:end)+Filters.PAItt(4,1:end),'b-','Linewidth',1.5);
    hold off
    xticks(dates(1:4:end))
        datetick('x',17,'keepticks');    
        ax = gca;
        labels = string(ax.XAxis.TickLabels);
        labels(1:5:end) = nan; 
        labels(2:5:end) = nan; 
        labels(3:5:end) = nan; 
        labels(4:5:end) = nan; 
        ax.XAxis.TickLabels = labels;    
        xlim([dates(1) dates(end)]);
        xtickangle(0)
        set(gca, 'Layer','top')
    axis tight
    ylim([0 1])
    box on
    legend('R-R Ext. Tally Index','Smoothed Prob','Filtered Prob','Units','inches','Position',[4 1.75 0.5 0.5])
print('-dpdf','..\Figures and Tables\Figure_5.pdf', '-fillpage')





%% -- Figure G.1: Measurement Errors -- %%

figure(101);
for ii = 1:NY
    subplot(NY,1,ii);
    hold on
    bar(dates,rrdates*3,1.02,'FaceColor',[0.75 0.75 0.75],'EdgeColor','none','ShowBaseLine','off')  
    bar(dates,rrdates*-3,1.02,'FaceColor',[0.75 0.75 0.75],'EdgeColor','none','ShowBaseLine','off')  
    h1 = plot(dates,(y(ii,:)-filtered_observables_average(ii,:))/GLOBALS.estimation.E(ii,ii),'k-','Linewidth',2);
    plot(dates,zeros(1,length(y)),'k--','Linewidth',1);
    plot(dates,[-2;2]*ones(1,length(y)),'k--','Linewidth',1);
    hold off
    xticks(dates(1:4:end))
        datetick('x',17,'keepticks');    
        ax = gca;
        labels = string(ax.XAxis.TickLabels);
        labels(1:5:end) = nan; 
        labels(2:5:end) = nan; 
        labels(3:5:end) = nan; 
        labels(4:5:end) = nan; 
        ax.XAxis.TickLabels = labels;    
        xlim([dates(1) dates(end)]);
        xtickangle(0)
        set(gca, 'Layer','top')
    axis tight
    ylim([-3 3])
    set(gca, 'Layer','top')

    box on
    title(GLOBALS.obslist{ii})
end
print('-dpdf','..\Figures and Tables\Figure_G1.pdf', '-fillpage')
